<?php

namespace Flow\FlowPayment\Controller\Payment;

class Init extends \Magento\Framework\App\Action\Action
{
    protected $logger;
    protected $helper;
    protected $cart;
    protected $checkoutSession;
    protected $resultJsonFactory;
    protected $quoteManagement;
    protected $storeManager;
    protected $configProvider;
    protected $productMetadata;
    protected $pluginVersion;

    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Checkout\Model\Cart $cart,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Controller\Result\JsonFactory $resultJsonFactory,
        \Magento\Quote\Model\QuoteManagement $quoteManagement,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Flow\FlowPayment\Model\Config\ConfigProvider $configProvider,
        \Psr\Log\LoggerInterface $logger,
        \Flow\FlowPayment\Helper\Helper $helper,
        \Magento\Framework\App\ProductMetadataInterface $productMetadata
    ) 
    {
        parent::__construct($context);

        $this->cart = $cart;
        $this->checkoutSession = $checkoutSession;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->quoteManagement = $quoteManagement;
        $this->storeManager = $storeManager;
        $this->configProvider = $configProvider;
        $this->logger = $logger;
        $this->helper = $helper;
        $this->productMetadata = $productMetadata;
        $this->pluginVersion = "3.0.10";
    }

    public function execute()
    {
        $this->logger->debug('Entering the init payment...');

        try {

            $config = $this->configProvider->getPluginConfig();
            $flowApi = $this->configProvider->getFlowApi();

            $order = $this->getOrder();

            //Changing the state/status of the order right away to pending payment
            $order->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT)
            ->setStatus(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT)
            ->addStatusToHistory(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT,  "Order waiting por payment");
            $order->save();

            $orderCurrency = $order->getOrderCurrencyCode();

            $this->checkoutSession->restoreQuote();
            $quote = $this->cart->getQuote();

            $guestEmail = isset($_GET['guestEmail']) ? $_GET['guestEmail'] : null;

            $orderAmount = $this->helper->getOrderAmount($order->getGrandTotal(), $orderCurrency);

            if ($guestEmail != null) {
                $quote->getBillingAddress()->setEmail($guestEmail);
                $quote->setData('customer_email', $quote->getBillingAddress()->getEmail());
                $quote->setData('customer_firstname', $quote->getBillingAddress()->getFirstName());
                $quote->setData('customer_lastname', $quote->getBillingAddress()->getLastName());
                $quote->setData('customer_is_guest', 1);
            }

            $quote->getPayment()->importData(['method' => \Flow\FlowPayment\Model\FlowPayment::CODE]);
            $quote->collectTotals()->save();

            $this->checkoutSession->setLastQuoteId($quote->getId());
            $this->checkoutSession->setLastSuccessQuoteId($quote->getId());
            $this->checkoutSession->setLastOrderId($order->getId());
            $this->checkoutSession->setLastRealOrderId($order->getIncrementId());
            $this->checkoutSession->setLastOrderStatus($order->getStatus());
            $this->checkoutSession->setGrandTotal($orderAmount);

            $baseUrl = $this->storeManager->getStore()->getBaseUrl();
            $returnUrl = $baseUrl.$config["RETURN_URL"];
            $confirmationUrl = $baseUrl.$config["CONFIRM_URL"];

            $subject = "Orden #".$order->getId(). " - ".$this->storeManager->getStore()->getName();

            $flowParams = [
                "commerce_order" => $order->getId(), 
                "subject" => $subject, 
                "currency" => $orderCurrency,
                "payment_currency" => $orderCurrency,
                "amount" => $orderAmount, 
                "email" => $order->getData("customer_email"), 
                "payment_method" => \Flow\FlowPayment\Model\FlowPayment::PAYMENT_METHOD, 
                "url_confirm" => $confirmationUrl, 
                "url_return" => $returnUrl,
                "customer" => $this->getCustomer($order)
            ];

            $shippingAddress = $this->getShippingAddress($order);
            if (!empty($shippingAddress)) {
                $flowParams["shipping_address"] = $shippingAddress;
            }

            $billingAddress = $this->getBillingAddress($order);
            if (!empty($billingAddress)) {
                $flowParams["billing_address"] = $billingAddress;
            }

            $items = $this->getItems($order);

            if (!empty($items)) {
                $flowParams["items"] = $items;
            }

            $flowParams["metadata"] = $this->getMetadataComerce();

            $this->logger->debug('Calling Flow service: order/create with params...', $flowParams);
            $result = $flowApi->order($flowParams);
            $this->logger->debug('Flow response: ', $result);

            $redirectUrl = $result["url_payment"];

            $jsonFactory = $this->resultJsonFactory->create();
            $jsonFactory->setData([
                'redirectUrl' => $redirectUrl
            ]);

            return $jsonFactory;

        } catch (\Exception $e) {

            $this->logger->error($e->getMessage());
            $jsonFactory = $this->resultJsonFactory->create();
            if ($e->getCode() === 1000) {
                $jsonFactory->setData([
                    'redirectUrl' => $this->_url->getUrl('flowpayment/error/index', ["message" => $e->getMessage()])
                ]);
            } else {
                $jsonFactory->setData([
                    'redirectUrl' => $this->_url->getUrl('flowpayment/error/index')
                ]);
            }

            return $jsonFactory;
        }
        
    }

    private function getOrder() {
        try {
            $orderId = $this->checkoutSession->getLastOrderId();
            if ($orderId == null) {
                return null;
            }
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            return $objectManager->create('\Magento\Sales\Model\Order')->load($orderId);
        } catch (\Exception $e) {
            return null;
        }
    }

    private function getCustomer(\Magento\Sales\Model\Order $order)
    {
        $customer = array(
            "id" => $order->getCustomerId() ?? null,
            "first_name" => $order->getCustomerFirstname(),
            "last_name" => $order->getCustomerLastname(),
            "email" => $order->getCustomerEmail(),
        );

        return $customer;
    }

    private function getShippingAddress(\Magento\Sales\Model\Order $order)
    {
        $data = $order->getShippingAddress()->getData();
        $address = array(
            "name" => $data["firstname"]." ".$data["lastname"],
            "country" => $data["country_id"],
            "line1" => $data["street"],
            "city" => $data["city"] ?? null,
            "zip" => $data["postcode"] ?? null,
            "phone" => $data["telephone"] ?? null
        );

        if (!empty($address["name"]) && !empty($address["country"]) && !empty($address["line1"])) {
            return $address;
        }

        return null;
    }

    private function getBillingAddress(\Magento\Sales\Model\Order $order)
    {
        $data = $order->getBillingAddress()->getData();
        $address = array(
            "name" => $data["firstname"]." ".$data["lastname"],
            "country" => $data["country_id"],
            "line1" => $data["street"],
            "city" => $data["city"] ?? null,
            "zip" => $data["postcode"] ?? null,
            "phone" => $data["telephone"] ?? null
        );

        if (!empty($address["name"]) && !empty($address["country"]) && !empty($address["line1"])) {
            return $address;
        }

        return null;
    }

    private function getItems(\Magento\Sales\Model\Order $order)
    {
        $items = $order->getAllItems();
        $itemsArray = array();

        foreach ($items as $item) {
            $data = $item->getData();
            $itemsArray[] = array(
                "type" => "sku",
                "sku" => $data["sku"] ??  null,
                "name" => $data["name"],
                "product_code" => $data["product_id"],
                "description" => $data["description"] ?? null,
                "unit_cost" => $this->helper->formatPrice($data["original_price"]) ?? null,
                "quantity" => (int) $data["qty_ordered"] ?? null,
                "amount" => $this->helper->formatPrice($data["price"]) ?? null,
            );
        }

        return $itemsArray;
    }

    /**
     * Return metadata information comerce plugin
     *
     * @return array
     */
    private function getMetadataComerce()
    {
        $version = $this->productMetadata->getVersion();
        $shopName = empty($this->storeManager->getStore()->getName()) ? "" : trim($this->storeManager->getStore()->getName());

        $metadata = array();
        $metadata[] = array("key" => 'ecommerce_name', "value" => "Magento", "visible" => false);
        $metadata[] = array("key" => 'ecommerce_version', "value" => $version, "visible" => false);
        $metadata[] = array("key" => 'plugin_name', "value" => "Flow Magento Checkout", "visible" => false);
        $metadata[] = array("key" => 'plugin_version', "value" => $this->pluginVersion, "visible" => false);
        if (strlen($shopName) > 0) {
            $metadata[] = array("key" => 'shop_name', "value" => $shopName, "visible" => false);
        }
        return $metadata;
    }

}
